"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteRestricao = exports.getOneRestricao = exports.getAllRestricoes = exports.createRestricao = void 0;
const express_validator_1 = require("express-validator");
const mongoose_1 = __importDefault(require("mongoose"));
const restricao_1 = __importDefault(require("../models/restricao"));
const createRestricao = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const { tipo, descricao } = req.body;
        const verifyRestricao = await restricao_1.default.findOne({ tipo });
        if (verifyRestricao) {
            const error = new Error(`A Restricao ${verifyRestricao.tipo} já foi criada! `);
            error.statusCode = 422;
            throw error;
        }
        const restricao = new restricao_1.default({
            tipo,
            descricao
        });
        await restricao.save();
        res.status(201).json({
            msg: 'Restricao criada com sucesso',
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createRestricao = createRestricao;
const getAllRestricoes = async (req, res, next) => {
    try {
        const restricoes = await restricao_1.default.find({});
        if (restricoes.length === 0) {
            return res.status(404).json({
                success: false,
                restricoes
            });
        }
        res.status(200).json({
            success: true,
            restricoes
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getAllRestricoes = getAllRestricoes;
const getOneRestricao = async (req, res, next) => {
    try {
        const id = req.params.id;
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (validId) {
            const restricao = await restricao_1.default.findOne({ _id: id });
            if (!restricao) {
                return res.status(404).json({
                    success: false,
                    restricao
                });
            }
            res.status(200).json({
                success: true,
                restricao
            });
        }
        else {
            return res.status(400).json({
                success: false,
                message: 'Id inválido.'
            });
        }
    }
    catch (err) {
        next(err);
    }
};
exports.getOneRestricao = getOneRestricao;
const deleteRestricao = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const id = req.params.id;
        // Verifica se o id é válido
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (!validId) {
            return res.status(400).json({
                success: false,
                message: 'ID inválido.',
            });
        }
        // Busca o Restricao para garantir que ele existe
        const restricao = await restricao_1.default.findById(id);
        if (!restricao) {
            return res.status(404).json({
                success: false,
                message: 'Restrição não encontrada.',
            });
        }
        // Deleta o Restricao
        await restricao_1.default.findByIdAndDelete(id);
        res.status(200).json({
            success: true,
            message: `Restricao ${restricao.tipo} deletada com sucesso.`,
        });
    }
    catch (err) {
        next(err); // Passa o erro para o middleware de erro
    }
};
exports.deleteRestricao = deleteRestricao;
